/** @file   fighter.cpp
 * @brief   Implementation of Fighter - class.
 * @version $Revision: 1.4 $
 * @author  Tomi Lamminsaari
 */

#include "fighter.h"
#include "GfxManager.h"
#include "settings.h"
#include "www_map.h"
#include "www_assert.h"
#include "fightercontroller.h"
#include "animplayer.h"
#include "redrawqueue.h"
#include "soundsamples.h"
#include "warglobals.h"
#include "gfxid.h"
#include "AnimId.h"

using std::string;
using std::ifstream;
using namespace eng2d;

namespace WeWantWar {

///
/// Static members, constants and datatypes
/// =======================================

const int Fighter::KGun1Nose;
const int Fighter::KGun2Nose;



///
/// Constructors, destructor and operators
/// ======================================
/** Constructor
 */
Fighter::Fighter( const string& aTypeName ) :
  GameObject(),
  iVelocity( 0 ),
  iGraphicUid( -1 ),
  iShadowUid( -1 ),
  iAttackSound( -1 )
{
  this->initFighter( aTypeName );
  
  m_pSSource = new SoundSource( "fighter", this->position() );
  Sound::createSoundSource( m_pSSource );
}



/** Destructor
 */
Fighter::~Fighter()
{
  /** We must not delete the Controller-object because the destructor of
   * GameObject - class will take care of that.
   */
   
  /** We must not delete the SoundSource here because the SoundPlayer -
   * class will take care of that.
   */
}




///
/// Public methods
/// ==============

/** Update method
 */
void Fighter::update()
{
  if ( this->state() == GameObject::STATE_KILLED ) {
    return;
  }
  
  if ( this->state() == GameObject::STATE_DYING ) {
    this->changeAngle( (rand() % 5) + 5 );
    
    Vec2D movevec( 0, -iVelocity );
    movevec.rotate( this->angle() );
    this->move( movevec );
    if ( this->altitude() < 0 ) {
      const Animation& explosionAnim = GameAnims::findAnimation(
          AnimId::KExplosionGrenade, 0 );
          
      AnimPlayer::spawn( explosionAnim, this->position(), 0 );
      Sound::playSample( SMP_GRENADE, false );
      this->hidden( true );
      this->state( GameObject::STATE_KILLED );
      
      float ac = 256.0 / 32.0;
      for (float a=0; a < 255; a += ac) {
        Vec2D spdVec( 0,-7 );
        spdVec.rotate( a );
        
        Bullet* b = BulletTable::createBullet( this, m_position,
                                               Bullet::EGrenade );
        b->iOwner = 0;
        b->setVelocity( spdVec );
        b->iRange = 120;
        b->iDamage = 20;
        WarGlobals::pBulletManager->spawnBullet( b );
      }
    }
    if ( static_cast<int>(this->altitude()) % 2 == 0 ) {
      // We create the flameanimation only every second update round. This
      // works as long as the altitude drops by odd values.
      const Animation flameAnim = GameAnims::findAnimation(
          AnimId::KFireFlameThrower, 0 );
      AnimPlayer::spawn( flameAnim, this->position(), 0 );
    }
    return;
  }
  
  // We are living
  BaseController* pC = this->getController();
  pC->update();
  if ( pC->forward() == 1 ) {
    Vec2D movevec( 0, -iVelocity );
    movevec.rotate( this->angle() );
    this->move( movevec );
  }
  this->changeAngle( pC->turn() );
  
  if ( pC->shoot() == 1 ) {
    this->shoot();
  }
  
  // Update the soudsource
  if ( m_pSSource != 0 ) {
    m_pSSource->position( this->position() );
  }
}



/** Kills this object
 */
void Fighter::kill()
{
  // Are we already killed or at least dying.
  if ( this->state() != GameObject::STATE_LIVING ) {
    return;
  }

  m_pSSource->turnOff();
  
  this->state( GameObject::STATE_DYING );
  this->verticalSpeed( -3 );
  Sound::playSample( SMP_ALIEN2_DIE, false );
}



/** Makes sound
 */
void Fighter::makeSound( GameObject::SoundID id ) const
{
  if ( id == GameObject::SND_ATTACK ) {
    if ( iAttackSound != -1 ) {
      Sound::playSample( iAttackSound, false );
    }
    
  } else if ( id == GameObject::SND_PAIN ) {
    Sound::playSample( SMP_METALHIT, false );
    
  }
}



/** Handles the bullet hits
 */
bool Fighter::hitByBullet( Bullet* pB )
{
  // Is this bullet shot by another fighter
  if ( pB->iOwner != 0 ) {
    if ( pB->iOwner->objectType() == ObjectID::TYPE_FIGHTER ) {
      return false;
    }
  }

  // Check if it's a crowbar
  if ( pB->iType == Bullet::ECrowbar ) {
    return false;
  }

  // cause damage
  this->causeDamage( pB );
  ParticleSparks* pP = new ParticleSparks( pB->iPosition, pB->velocity(), 12 );
  WarGlobals::pPartManager->addSystem( pP );
  return true;
}



/** Redraws this object
 */
void Fighter::redraw( RedrawQueue* pQueue )
{
  if ( this->hidden() == true ) {
    return;
  }
  if ( this->state() == GameObject::STATE_KILLED ) {
    return;
  }

  int index = this->angle() / 4;
  BITMAP* fighterGraphic = GfxManager::findBitmap( iGraphicUid );
  BITMAP* shadowGraphic = GfxManager::findBitmap( iShadowUid, index );
  
  Vec2D p( this->position() );
  int x = static_cast<int>( p.x() ) - Map::scrollX;
  int y = static_cast<int>( p.y() ) - Map::scrollY;
  int alt = static_cast<int>( this->altitude() / 5.0 );
  int shaX = x - alt;
  int shaY = y + alt;
  if ( shadowGraphic != 0 ) {
    shaX -= shadowGraphic->w / 2;
    shaY -= shadowGraphic->h / 2;
  }
  
  x -= fighterGraphic->w / 2;
  y -= fighterGraphic->h / 2;
  
  if ( shadowGraphic != 0 ) {
    pQueue->add( RedrawQueue::PRI_ABOVE_NORMAL, shaX,shaY,
                 RedrawQueue::SPTYPE_SPRITE, RedrawQueue::BMODE_MULTIPLY,
                 128, shadowGraphic );
  }
  pQueue->addRotatedSprite( RedrawQueue::PRI_FLYING_HIGH, x,y,
                            itofix(this->angle()), fighterGraphic );
}



/** Moves this fighter
 */
bool Fighter::move( const Vec2D& rMoveVec )
{
  Vec2D tmp( this->position() );
  tmp += rMoveVec;
  this->position( tmp );
  return false;
}




///
/// Getter methods
/// ==============

/** Tells if we're reloading
 */
bool Fighter::reloading() const
{
  if ( this->getCounter(0) < 0 ) {
    return false;
  }
  return true;
}



/** Returns the type of this object.
 */
ObjectID::Type Fighter::objectType() const
{
  return ObjectID::TYPE_FIGHTER;
}




///
/// Private or Protected methods
/// ============================

/** Initializes this fighter.
 */
int Fighter::initFighter( const string& aPrefix )
{
  ObjectID::Type oid = ObjectID::TYPE_FIGHTER;
  iVelocity = Settings::floatObjProp(oid, aPrefix + "_spd:");
  iGraphicUid = Settings::intObjProp(oid, aPrefix + "_gfxId:");
  iShadowUid = Settings::intObjProp(oid, aPrefix + "_shadowId:");
  iBulletType = static_cast<Bullet::TType>( Settings::intObjProp(oid, aPrefix + "_bulletId:") );
  iAttackSound = Settings::intObjProp(oid, aPrefix + "_sndAttack:");
  this->boundingSphere( Settings::floatObjProp(oid, aPrefix + "_bsphere:") );
  this->setArmor( Settings::floatObjProp(oid, aPrefix + "_armor:") );
  this->altitude( Settings::floatObjProp(oid, aPrefix + "_alt:") );
  Vec2D gunPos1( Settings::floatObjProp(oid, aPrefix + "_gunX1:"),
                 Settings::floatObjProp(oid, aPrefix + "_gunY1:") );
  Vec2D gunPos2( Settings::floatObjProp(oid, aPrefix + "_gunX2:"),
                 Settings::floatObjProp(oid, aPrefix + "_gunY2:") );
  this->addCtrlPoint( gunPos1 );
  this->addCtrlPoint( gunPos2 );
  
  int attackDist = Settings::intObjProp(oid, aPrefix + "_attackDistance:");
  int attackLen = Settings::intObjProp(oid, aPrefix + "_attackDuration:");
  int evadeLen = Settings::intObjProp(oid, aPrefix + "_evadeLength:");
  FighterController* controller = new FighterController(this);
  controller->setAttackPhaseLength(attackLen);
  controller->setEvadePhaseLength(evadeLen);
  controller->setAttackDistance(attackDist);
  this->setController( controller );
  return 0;
}

void Fighter::shoot()
{
  if ( this->reloading() == false ) {
    // Shoot
    Vec2D gunV1( this->getCtrlPoint( KGun1Nose ) );
    Vec2D gunV2( this->getCtrlPoint( KGun2Nose ) );
    gunV1 += this->position();
    gunV2 += this->position();
    
    Bullet* bullet1 = BulletTable::createBullet( this, gunV1, iBulletType );
    Bullet* bullet2 = BulletTable::createBullet( this, gunV2, iBulletType );
    
    WarGlobals::pBulletManager->spawnBullet( bullet1 );
    WarGlobals::pBulletManager->spawnBullet( bullet2 );
    
    this->makeSound( GameObject::SND_ATTACK );
    this->setCounter( 0, 2 );
  }
}

} // end of namespace
